# Generating user certificates

- The most popular ways to authenticate users with Kubernetes are:

  - TLS certificates

  - JSON Web Tokens (OIDC or ServiceAccount tokens)

- We're going to see how to use TLS certificates

- We will generate a certificate for an user and give them some permissions

- Then we will use that certificate to access the cluster

---

## Heads up!

- The demos in this section require that we have access to our cluster's CA

- This is easy if we are using a cluster deployed with `kubeadm`

- Otherwise, we may or may not have access to the cluster's CA

- We may or may not be able to use the CSR API instead

---

## Check that we have access to the CA

- Make sure that you are logged on the node hosting the control plane

  (if a cluster has been provisioned for you for a training, it's `node1`)

.exercise[

- Check that the CA key is here:
  ```bash
  sudo ls -l /etc/kubernetes/pki
  ```

]

The output should include `ca.key` and `ca.crt`.

---

## How it works

- The API server is configured to accept all certificates signed by a given CA

- The certificate contains:

  - the user name (in the `CN` field)

  - the groups the user belongs to (as multiple `O` fields)

.exercise[

- Check which CA is used by the Kubernetes API server:
  ```bash
  sudo grep crt /etc/kubernetes/manifests/kube-apiserver.yaml
  ```

]

This is the flag that we're looking for:
```
--client-ca-file=/etc/kubernetes/pki/ca.crt
```

---

## Generating a key and CSR for our user

- These operations could be done on a separate machine

- We only need to transfer the CSR (Certificate Signing Request) to the CA

  (we never need to expoes the private key)

.exercise[

- Generate a private key:
  ```bash
  openssl genrsa 4096 > user.key
  ```

- Generate a CSR:
  ```bash
  openssl req -new -key user.key -subj /CN=jerome/O=devs/O=ops > user.csr
  ```

]

---

## Generating a signed certificate

- This has to be done on the machine holding the CA private key

  (copy the `user.csr` file if needed)

.exercise[

- Verify the CSR paramters:
  ```bash
  openssl req -in user.csr  -text | head
  ```

- Generate the certificate:
  ```bash
  sudo openssl x509 -req \
      -CA /etc/kubernetes/pki/ca.crt -CAkey /etc/kubernetes/pki/ca.key \
      -in user.csr -days 1 -set_serial 1234 > user.crt
  ```

]

If you are using two separate machines, transfer `user.crt` to the other machine.

---

## Adding the key and certificate to kubeconfig

- We have to edit our `.kube/config` file

- This can be done relatively easily with `kubectl config`

.exercise[

- Create a new `user` entry in our `.kube/config` file:
  ```bash
  kubectl config set-credentials jerome \
      --client-key=user.key --client-certificate=user.crt 
  ```

]

The configuration file now points to our local files.

We could also embed the key and certs with the `--embed-certs` option.

(So that the kubeconfig file can be used without `user.key` and `user.crt`.)

---

## Using the new identity

- At the moment, we probably use the admin certificate generated by `kubeadm`

  (with `CN=kubernetes-admin` and `O=system:masters`)

- Let's edit our *context* to use our new certificate instead!

.exercise[

- Edit the context:
  ```bash
  kubectl config set-context --current --user=jerome 
  ```

- Try any command:
  ```bash
  kubectl get pods
  ```

]

Access will be denied, but we should see that were correctly *authenticated* as `jerome`.

---

## Granting permissions

- Let's add some read-only permissions to the `devs` group (for instance)

.exercise[

- Switch back to our admin identity:
  ```bash
  kubectl config set-context --current --user=kubernetes-admin 
  ```

- Grant permissions:
  ```bash
  kubectl create clusterrolebinding devs-can-view \
      --clusterrole=view --group=devs
  ```

]

---

## Testing the new permissions

- As soon as we create the ClusterRoleBinding, all users in the `devs` group get access

- Let's verify that we can e.g. list pods!

.exercise[

- Switch to our user identity again:
  ```bash
  kubectl config set-context --current --user=jerome
  ```

- Test the permissions:
  ```bash
  kubectl get pods
  ```

]

???

:EN:- Authentication with user certificates
:FR:- Identification par certificat TLS
